/*

maxnativeskin.cpp

MAX native skin & bones stuff should be migrated here

*/

#include "stdafx.h"
#include "DffExp.h"
#include "modstack.h"
#include "maxnativeskin.h"
#include "matrixmangle.h"
#include "utilities.h"
#undef IDC_STATIC   //avoid annoying warning generated by bonesdef.h
#include "..\SAMPLES\Modifier\BonesDef\bonesdef.h"

Modifier * 
FindBonesDefModifier (INode* nodePtr)
{
    return FindModifier( nodePtr, Class_ID(9815843,87654) );
}

INode * 
FindFirstMaxSkinNode(INode* nodePtr)
//Return a pointer to the first node with a max native skin modifier in the hierarchy
{
    int i;
    INode *skinPtr;

    if (FindBonesDefModifier( nodePtr ))
    {
        return nodePtr;
    }

    for (i=0; i<nodePtr->NumberOfChildren(); i++) 
    {
        skinPtr=FindFirstMaxSkinNode(nodePtr->GetChildNode(i));
        if (skinPtr)
        {
            return skinPtr;
        }
    }

	/* Not found. */
	return NULL;
}


bool
NodeContainsBones(INode* Node)
//Return true if there is at least one bone node in this hierarchy
{
	int i;
	
	Object* ObjectPtr = Node->GetObjectRef();
	if (ObjectPtr) {
		if ((ObjectPtr->SuperClassID()==HELPER_CLASS_ID)
			  &&(ObjectPtr->ClassID()==Class_ID(BONE_CLASS_ID,0))) {
			return true;
		}
	}
	
    for (i=0; i<Node->NumberOfChildren(); i++)
        if (NodeContainsBones(Node->GetChildNode(i))) return true;

	// Not found.
	return false;
}

/*****************************************************************************
 *
 *      Max native skinning functions (bonesDefModifier)
 *
 ****************************************************************************/


class bonesDefModContextEnumProc : public ModContextEnumProc
{        
	public:
        BoneModData *modData;
		bonesDefModContextEnumProc() { this->modData = NULL; }
		BOOL proc(ModContext *mc);  // Return FALSE to stop, TRUE to continue.
};

BOOL
bonesDefModContextEnumProc::proc(ModContext *mc)
{
    modData = (BoneModData *)mc->localData;
    return TRUE;
}

void
DFFExport::CalculateMaxSkinVertexBoneMap(INode *node, RwInt32 *vertexMap)
//On exit
//  skinVertexIndices gives the indices of up to 4 bones that influence a vertex in the skin
//      packed as 8 bits into a UInt32 (is anyone asserting that # bones is < 256 BTW?!)
//  skinMatrixWeights gives the corresponding weights for each bone's influence on each skin vertex,
//      normalized s.t. their sum is one.
{
    int i, j, k;

    skinMatrixWeights = (RwMatrixWeights *)RwMalloc(sizeof(RwMatrixWeights) * m_remapper.GetNumVertices());
    skinVertexIndices = (RwUInt32 *)RwMalloc(sizeof(RwUInt32) * m_remapper.GetNumVertices());
                    
    /* now get data from bonesDef modifier  */    
    BonesDefMod *bonesDefMod = (BonesDefMod *)FindBonesDefModifier(node);    
    
    if (bonesDefMod)    
    {
        /* Get access to bone info */
        BoneDataClassList *boneList = &bonesDefMod->BoneData;
        int numBones = boneList->Count();

        /* Get access to vertex info */
        bonesDefModContextEnumProc enumerator;
        bonesDefMod->EnumModContexts(&enumerator);
                
        //for each skin vertex
        for (i = 0; i < m_remapper.GetNumVertices(); i++)
        {                                
            int vertexIndex = m_remapper.GetVertex(i)->vertexIndex;
            VertexListClass* vData = enumerator.modData->VertexData[vertexIndex];
            int numWeightsSet = 0;
            RwBool moreThan4Weights = FALSE;

            skinVertexIndices[vertexMap[i]] = 0;
            skinMatrixWeights[vertexMap[i]].w0 = 0.0f;
            skinMatrixWeights[vertexMap[i]].w1 = 0.0f;
            skinMatrixWeights[vertexMap[i]].w2 = 0.0f;
            skinMatrixWeights[vertexMap[i]].w3 = 0.0f;

            //for each bone influencing this skin vertex
            for (j = 0; j < vData->d.Count(); j++)
            {
                float thisInf = vData->d[j].Influences;
                int thisBone = vData->d[j].Bones;

                if (thisInf != 0.0f)
                {
                    //find the corresponding Renderware bone
                    //! bit slow for meshes with lots of vertices & bones
                    //could be much faster to store a pointer to the Renderware bone via
                    //INode's user prop buffer.
                    BoneDataClass bone = (*boneList)[thisBone];
                    for (k=0; k < CSNumBones; k++)
                    {
                        if (CSBoneIndexNodePointers[k] == bone.Node)
                        {
                            //Add the MAX bone's index & weight to Renderware bone
                            if (numWeightsSet < 4)
                            {
                                skinVertexIndices[vertexMap[i]] |= k << (numWeightsSet*8);
                                void *weightArrayPtr = &(skinMatrixWeights[vertexMap[i]]);
                                RwReal *weightArray = (RwReal *)weightArrayPtr;
                                weightArray[numWeightsSet] = thisInf;
                                numWeightsSet++;
                            }
                            else
                            {
                                moreThan4Weights = TRUE;
                            }                        
                        }
                    }
                }
            }    
            /* If there were more than 4 weights on this vertex re-scale the first 4. Should really pick the
               4 highest and re-scale */
            if (numWeightsSet == 0)
            {
                /* no weights so ensure at least one weight is set. Find a bone
                   based upon other vertices used by triangles using this vertex,
                   might be better to do a link by proximity */
                int tri;
                RwBool copied = FALSE;
                
                /* First find a triangle using this vertex which has a weight
                   set on one of it's other vertices */
                for (tri=0; tri<m_remapper.GetNumFaces() && !copied; tri++)
                {
                    ReMapperFaceLoop *face = m_remapper.GetFace(tri);

                    if (face->index[0] == i ||
                        face->index[1] == i ||
                        face->index[2] == i)
                    {
                        int indextocopy = -1;
                        if (skinMatrixWeights[vertexMap[face->index[0]]].w0 > 0.0f)
                        {
                            indextocopy = 0;                            
                        }
                        else if (skinMatrixWeights[vertexMap[face->index[1]]].w0 > 0.0f)
                        {
                            indextocopy = 1;                            
                        }
                        else if (skinMatrixWeights[vertexMap[face->index[2]]].w0 > 0.0f)
                        {
                            indextocopy = 2;                            
                        }

                        if (indextocopy != -1)
                        {
                            skinMatrixWeights[vertexMap[i]].w0 = skinMatrixWeights[vertexMap[face->index[indextocopy]]].w0;
                            skinMatrixWeights[vertexMap[i]].w1 = skinMatrixWeights[vertexMap[face->index[indextocopy]]].w1;
                            skinMatrixWeights[vertexMap[i]].w2 = skinMatrixWeights[vertexMap[face->index[indextocopy]]].w2;
                            skinMatrixWeights[vertexMap[i]].w3 = skinMatrixWeights[vertexMap[face->index[indextocopy]]].w3;
                            skinVertexIndices[vertexMap[i]] = skinVertexIndices[vertexMap[face->index[indextocopy]]];
                            copied = TRUE;
                        }
                    }
                }
            }
            
            //normalize all the weights s.t. they sum to 1
            if (numWeightsSet > 0)
            {
                RwReal weightScale = 1.0f / (skinMatrixWeights[vertexMap[i]].w0+skinMatrixWeights[vertexMap[i]].w1+
                                             skinMatrixWeights[vertexMap[i]].w2+skinMatrixWeights[vertexMap[i]].w3);
                skinMatrixWeights[vertexMap[i]].w0 *= weightScale;
                skinMatrixWeights[vertexMap[i]].w1 *= weightScale;
                skinMatrixWeights[vertexMap[i]].w2 *= weightScale;
                skinMatrixWeights[vertexMap[i]].w3 *= weightScale;
            }
        }  
    }                         
}

void
DFFExport::ProcessMaxSkinNode(INode *node, RpClump *clump)
{
    if (!FindBonesDefModifier(node))
    {
        //not a skin, forget it
        return;
    }

    //we need to transform the skin's vertices into the correct space relative to the bones.
    //work out a matrix to do that.
    Matrix3 transform;

    Matrix3 skinMat = node->GetObjTMAfterWSM(m_tvStart);
    
    INode *boneNode=FindMaxBonesHierarchy(node);
    
    //don't ask why, it just works. :-P
    if (boneNode->GetParentNode()->IsRootNode())
    {
        //bone is top of hierarchy
        Matrix3 boneMat = boneNode->GetObjTMAfterWSM(m_tvStart);
        transform = Inverse(boneMat) *  skinMat;
    }
    else
    {
        //bone is a child of something else
        transform = skinMat * Inverse( FindTopOfHierarchy( boneNode )->GetObjTMAfterWSM(m_tvStart) );
    }

    //don't forget to add user's scaling to skin vertices:
    transform *= m_scaleMatrix;

    //get vertices & normals of skin from MAX
    RwInt32 *vertexMap = NULL;
    RpAtomic *atomic = NULL;
    ExtractGeometryFromNode( node, clump, RpClumpGetFrame(clump), atomic, vertexMap, transform );

    if (vertexMap)
    {
        // create a vertex bone map - result stuffed into DFFExport::vertexBoneMap data member
        CalculateMaxSkinVertexBoneMap(node, vertexMap);  

        // get rid of the vertex map
        RwFree(vertexMap);
    }

    if (atomic && vertexMap)
    {
        RpSkin *skin = RpSkinCreate(atomic, CSNumBones, skinMatrixWeights, skinVertexIndices, skinInverseMatrices, skinFlags);                
        if (skin)
        {
            int i;
            // set the bonetags to match the frame hierarchy
            for (i=0; i<CSNumBones; i++)
            {                    
                skin->pBoneInfo[i].boneTag = CSBoneIndexTags[i];
            }
        }
    
        RwFree(skinMatrixWeights);
        skinMatrixWeights = NULL;
        RwFree(skinVertexIndices);
        skinVertexIndices = NULL;
    }
}

void
DFFExport::ProcessMaxSkinNodesInHierarchy(INode *node, RpClump *clump)
{
    int numChildren, i;

    ProcessMaxSkinNode(node, clump);

    numChildren = node->NumberOfChildren();
    for (i=0; i<numChildren; i++)
    {
        INode *childNode = node->GetChildNode(i);
        ProcessMaxSkinNodesInHierarchy(childNode, clump);    
    }
}

INode *
FindMaxBonesHierarchy(INode *skinNode)
//Return the hierarchy containing the first bone owned by skinNode
{
    /* now get data from physique */    
    BonesDefMod *bonesDefMod = (BonesDefMod *)FindBonesDefModifier(skinNode);    
    
    if (bonesDefMod)    
    {
        /* Get access to bone info */
        BoneDataClassList *boneList = &bonesDefMod->BoneData;
        BoneDataClass bone = (*boneList)[0];
        INode *boneHierarchy = bone.Node;

        return boneHierarchy;
    }
    else
    {
        /* no bones */
        return skinNode;
    }
}

RwExpError FindMaxSkinAndBonesNodes(
    INode *sceneRoot,               //input - root of entire MAX scene.
    INode *topOfSelectedHierarchy,  //input - highest node in heirarchy user selected.
    INode * &maxSkinNode,           //output - node with a max skin modifier with some relation to selected node.
    INode * &topOfMaxBonesNodes     //output - node with max bones for that skin.
    )
{
    if (FindBonesDefModifier(topOfSelectedHierarchy))
    {
        //This is the skin node, so locate the root of the tree containing the bones
        //which this skin references.
        maxSkinNode=topOfSelectedHierarchy;

        INode *bonesNode = FindMaxBonesHierarchy(maxSkinNode);

        if (bonesNode == 0)
        {
            return RwExpError("A valid skin modifier was selected, but no bone nodes could be found");
        }

        topOfMaxBonesNodes = FindTopOfHierarchy(bonesNode);
    }
    else if (NodeContainsBones(topOfSelectedHierarchy))
    {
        //The bone node has been selected
        //Make sure that this is the root of the bones tree
        topOfMaxBonesNodes = FindTopOfHierarchy(topOfSelectedHierarchy);

        //Now find and select the first skin node for export
        maxSkinNode=FindFirstMaxSkinNode(sceneRoot);

        if (maxSkinNode == 0)
        {
            return RwExpError("A valid bone was selected, but no skin could be found!");
        }

        //Make sure that this is the top of the hierarchy
        maxSkinNode = FindTopOfHierarchy(maxSkinNode);
    }
    else
    {
        //error - need to select either the bones or the skin
        return RwExpError("The skin or one of the bones must be selected for 3D studio max native skinning");
    }

    //okay.
    return RwExpError("");
}
